/**
 * Script d'affichage des distances - Solution robuste
 * Ce script remplace les précédents scripts de correction des distances
 */

document.addEventListener('DOMContentLoaded', function() {
  console.log("=== SCRIPT D'AFFICHAGE DES DISTANCES CHARGÉ ===");
  
  // Fonction pour afficher un message dans la console avec un préfixe
  function logMessage(message, type = 'info') {
    const prefix = "[DISTANCE-DISPLAY]";
    if (type === 'error') {
      console.error(prefix, message);
    } else if (type === 'warn') {
      console.warn(prefix, message);
    } else {
      console.log(prefix, message);
    }
  }
  
  // Fonction pour mettre à jour l'affichage d'une distance
  function updateDistanceDisplay(id, distanceKm) {
    try {
      // Formater la distance avec 1 décimale
      const formattedDistance = parseFloat(distanceKm).toFixed(1);
      logMessage(`Mise à jour de la distance pour ID ${id}: ${formattedDistance} km`);
      
      // Rechercher les éléments de distance pour cet ID
      const distanceContainer = document.getElementById(`distance-${id}`);
      const gridDistanceContainer = document.getElementById(`grid-distance-${id}`);
      
      // Mettre à jour le conteneur de distance dans la vue liste
      if (distanceContainer) {
        logMessage(`Conteneur de distance trouvé pour ID ${id}`);
        
        // Remplacer complètement le contenu HTML
        distanceContainer.innerHTML = `
          <p class="distance-value" style="color: #ff0000 !important; font-weight: bold !important;">${formattedDistance}</p>
          <p class="distance-unit">km</p>
        `;
        
        // Forcer l'affichage et le style
        distanceContainer.style.display = 'flex';
        distanceContainer.style.backgroundColor = '#e9f5dc';
        distanceContainer.classList.add('road-distance');
        
        logMessage(`Distance mise à jour dans la vue liste pour ID ${id}`);
      } else {
        logMessage(`Conteneur de distance non trouvé pour ID ${id} dans la vue liste`, 'warn');
      }
      
      // Mettre à jour le conteneur de distance dans la vue grille
      if (gridDistanceContainer) {
        logMessage(`Conteneur de distance de grille trouvé pour ID ${id}`);
        
        // Mettre à jour le contenu
        const valueSpan = gridDistanceContainer.querySelector('.grid-distance-value');
        if (valueSpan) {
          valueSpan.textContent = formattedDistance;
          valueSpan.style.color = '#ff0000';
          valueSpan.style.fontWeight = 'bold';
        } else {
          // Si l'élément n'existe pas, remplacer tout le contenu
          gridDistanceContainer.innerHTML = `
            <span class="grid-distance-value" style="color: #ff0000; font-weight: bold;">${formattedDistance}</span>
            <span class="grid-distance-unit">km</span>
          `;
        }
        
        // Forcer l'affichage
        gridDistanceContainer.style.display = 'flex';
        gridDistanceContainer.style.backgroundColor = '#e9f5dc';
        
        logMessage(`Distance mise à jour dans la vue grille pour ID ${id}`);
      } else {
        logMessage(`Conteneur de distance non trouvé pour ID ${id} dans la vue grille`, 'warn');
      }
      
      // Si aucun conteneur n'a été trouvé, essayer de créer de nouveaux éléments
      if (!distanceContainer && !gridDistanceContainer) {
        logMessage(`Aucun conteneur trouvé pour ID ${id}, tentative de création`, 'warn');
        
        // Rechercher l'élément farmer-item par son attribut data-id
        const farmerItems = document.querySelectorAll(`.farmer-item[data-id="${id}"], .grid-item[data-id="${id}"]`);
        
        farmerItems.forEach(item => {
          logMessage(`Élément trouvé pour ID ${id}: ${item.className}`);
          
          // Vérifier si c'est un élément de liste ou de grille
          if (item.classList.contains('farmer-item')) {
            // Créer un nouveau conteneur de distance pour la vue liste
            const newDistanceContainer = document.createElement('div');
            newDistanceContainer.className = 'farmer-distance road-distance';
            newDistanceContainer.id = `distance-${id}`;
            newDistanceContainer.style.display = 'flex';
            newDistanceContainer.style.backgroundColor = '#e9f5dc';
            
            newDistanceContainer.innerHTML = `
              <p class="distance-value" style="color: #ff0000 !important; font-weight: bold !important;">${formattedDistance}</p>
              <p class="distance-unit">km</p>
            `;
            
            // Trouver où insérer le nouvel élément
            const content = item.querySelector('.farmer-content');
            if (content) {
              const farmerInfo = content.querySelector('.farmer-info');
              if (farmerInfo && farmerInfo.nextElementSibling) {
                content.insertBefore(newDistanceContainer, farmerInfo.nextElementSibling);
              } else {
                content.appendChild(newDistanceContainer);
              }
              logMessage(`Nouveau conteneur de distance créé pour la vue liste, ID ${id}`);
            }
          } else if (item.classList.contains('grid-item')) {
            // Créer un nouveau conteneur de distance pour la vue grille
            const imageContainer = item.querySelector('.grid-item-image-container');
            if (imageContainer) {
              const newGridDistanceContainer = document.createElement('div');
              newGridDistanceContainer.className = 'grid-item-distance';
              newGridDistanceContainer.id = `grid-distance-${id}`;
              newGridDistanceContainer.style.display = 'flex';
              newGridDistanceContainer.style.backgroundColor = '#e9f5dc';
              
              newGridDistanceContainer.innerHTML = `
                <span class="grid-distance-value" style="color: #ff0000; font-weight: bold;">${formattedDistance}</span>
                <span class="grid-distance-unit">km</span>
              `;
              
              imageContainer.appendChild(newGridDistanceContainer);
              logMessage(`Nouveau conteneur de distance créé pour la vue grille, ID ${id}`);
            }
          }
        });
      }
      
      // Mettre à jour l'attribut data-distance pour le tri
      farmerItems = document.querySelectorAll(`.farmer-item[data-id="${id}"], .grid-item[data-id="${id}"]`);
      farmerItems.forEach(item => {
        item.setAttribute('data-distance', distanceKm);
      });
      
      return true;
    } catch (error) {
      logMessage(`Erreur lors de la mise à jour de la distance pour ID ${id}: ${error.message}`, 'error');
      console.error(error);
      return false;
    }
  }
  
  // Fonction pour intercepter les appels fetch à l'API de distance
  function interceptFetchCalls() {
    logMessage("Interception des appels fetch");
    
    // Sauvegarder la fonction fetch originale
    const originalFetch = window.fetch;
    
    // Remplacer la fonction fetch par notre version
    window.fetch = function(url, options) {
      // Vérifier si l'URL concerne notre API de distance
      if (url && typeof url === 'string' && url.includes('/get-distance.php')) {
        logMessage(`Appel API détecté: ${url}`);
        
        // Extraire l'ID de l'élément à partir des attributs data-* du bouton actif
        let currentItemId = null;
        const activeButton = document.querySelector('.geolocation-button[disabled]');
        if (activeButton) {
          const moduleId = activeButton.id.replace('geolocate-button-', '');
          const farmerList = document.getElementById(`farmer-list-${moduleId}`);
          if (farmerList) {
            // Trouver tous les éléments farmer-item
            const items = farmerList.querySelectorAll('.farmer-item');
            items.forEach(item => {
              const coords = item.getAttribute('data-coordinates');
              if (coords && url.includes(encodeURIComponent(coords.split(',').reverse().join(',')))) {
                currentItemId = item.getAttribute('data-id');
                logMessage(`ID trouvé pour l'appel API: ${currentItemId}`);
              }
            });
          }
        }
        
        // Appeler la fonction fetch originale et intercepter la réponse
        return originalFetch(url, options).then(response => {
          // Cloner la réponse pour pouvoir la lire
          return response.clone().text().then(text => {
            try {
              // Parser la réponse JSON
              const data = JSON.parse(text);
              
              // Vérifier si la réponse contient une distance
              if (data.features && 
                  data.features[0] && 
                  data.features[0].properties && 
                  data.features[0].properties.summary && 
                  typeof data.features[0].properties.summary.distance === 'number') {
                
                // Extraire la distance en kilomètres
                const distanceKm = data.features[0].properties.summary.distance / 1000;
                logMessage(`Distance extraite de la réponse: ${distanceKm.toFixed(1)} km`);
                
                // Si nous avons un ID, mettre à jour l'affichage
                if (currentItemId) {
                  updateDistanceDisplay(currentItemId, distanceKm);
                } else {
                  logMessage("ID non trouvé pour cette réponse", 'warn');
                  
                  // Essayer de trouver l'ID à partir des paramètres de l'URL
                  const urlParams = new URL(url, window.location.origin).searchParams;
                  const endCoords = urlParams.get('end');
                  if (endCoords) {
                    logMessage(`Recherche de l'élément avec les coordonnées: ${endCoords}`);
                    
                    // Convertir les coordonnées au format attendu par les attributs data-coordinates
                    const [endLng, endLat] = endCoords.split(',');
                    const dataCoords = `${endLat},${endLng}`;
                    
                    // Rechercher l'élément avec ces coordonnées
                    const items = document.querySelectorAll('[data-coordinates]');
                    items.forEach(item => {
                      const itemCoords = item.getAttribute('data-coordinates');
                      if (itemCoords === dataCoords) {
                        const itemId = item.getAttribute('data-id');
                        if (itemId) {
                          logMessage(`ID trouvé via les coordonnées: ${itemId}`);
                          updateDistanceDisplay(itemId, distanceKm);
                        }
                      }
                    });
                  }
                }
              } else if (data.error) {
                logMessage(`Erreur API: ${data.error}`, 'error');
              }
            } catch (error) {
              logMessage(`Erreur lors du traitement de la réponse: ${error.message}`, 'error');
            }
            
            // Retourner la réponse originale
            return response;
          }).catch(error => {
            logMessage(`Erreur lors de la lecture de la réponse: ${error.message}`, 'error');
            return response;
          });
        });
      }
      
      // Pour les autres URL, utiliser la fonction fetch originale
      return originalFetch(url, options);
    };
    
    logMessage("Interception des appels fetch configurée");
  }
  
  // Fonction pour observer les modifications du DOM
  function observeDOMChanges() {
    logMessage("Configuration de l'observateur de mutations");
    
    // Créer un observateur de mutations
    const observer = new MutationObserver(function(mutations) {
      mutations.forEach(function(mutation) {
        // Vérifier si des nœuds ont été ajoutés
        if (mutation.addedNodes.length > 0) {
          mutation.addedNodes.forEach(function(node) {
            // Vérifier si le nœud est un élément HTML
            if (node.nodeType === Node.ELEMENT_NODE) {
              // Vérifier si c'est un conteneur de distance
              if (node.classList && 
                  (node.classList.contains('farmer-distance') || 
                   node.classList.contains('grid-item-distance'))) {
                
                logMessage("Nouveau conteneur de distance détecté");
                
                // Extraire l'ID
                let id = null;
                if (node.id) {
                  id = node.id.replace('distance-', '').replace('grid-distance-', '');
                } else {
                  // Essayer de trouver l'ID à partir du parent
                  const parent = node.closest('[data-id]');
                  if (parent) {
                    id = parent.getAttribute('data-id');
                  }
                }
                
                if (id) {
                  logMessage(`ID du nouveau conteneur: ${id}`);
                  
                  // Vérifier si le parent a un attribut data-distance
                  const parent = node.closest('[data-distance]');
                  if (parent) {
                    const distance = parseFloat(parent.getAttribute('data-distance'));
                    if (!isNaN(distance) && distance > 0) {
                      logMessage(`Distance trouvée dans l'attribut data-distance: ${distance}`);
                      updateDistanceDisplay(id, distance);
                    }
                  }
                }
              }
            }
          });
        }
      });
    });
    
    // Observer tout le document
    observer.observe(document.body, {
      childList: true,
      subtree: true
    });
    
    logMessage("Observateur de mutations configuré");
  }
  
  // Fonction pour corriger les distances existantes
  function fixExistingDistances() {
    logMessage("Correction des distances existantes");
    
    // Rechercher tous les éléments avec un attribut data-distance
    const itemsWithDistance = document.querySelectorAll('[data-distance]');
    logMessage(`Nombre d'éléments avec data-distance: ${itemsWithDistance.length}`);
    
    itemsWithDistance.forEach(item => {
      const id = item.getAttribute('data-id');
      const distance = parseFloat(item.getAttribute('data-distance'));
      
      if (id && !isNaN(distance) && distance > 0) {
        logMessage(`Élément trouvé avec ID ${id} et distance ${distance}`);
        updateDistanceDisplay(id, distance);
      }
    });
    
    // Rechercher tous les conteneurs de distance
    const distanceContainers = document.querySelectorAll('.farmer-distance, .grid-item-distance');
    logMessage(`Nombre de conteneurs de distance: ${distanceContainers.length}`);
    
    distanceContainers.forEach(container => {
      // Extraire l'ID
      let id = null;
      if (container.id) {
        id = container.id.replace('distance-', '').replace('grid-distance-', '');
      } else {
        // Essayer de trouver l'ID à partir du parent
        const parent = container.closest('[data-id]');
        if (parent) {
          id = parent.getAttribute('data-id');
        }
      }
      
      if (id) {
        // Rechercher la valeur de distance
        const valueElement = container.querySelector('.distance-value, .grid-distance-value');
        if (valueElement) {
          const distance = parseFloat(valueElement.textContent);
          if (!isNaN(distance) && distance > 0) {
            logMessage(`Conteneur trouvé avec ID ${id} et distance ${distance}`);
            updateDistanceDisplay(id, distance);
          }
        }
      }
    });
    
    logMessage("Correction des distances existantes terminée");
  }
  
  // Fonction pour ajouter un gestionnaire d'événements au bouton de géolocalisation
  function setupGeolocationButton() {
    logMessage("Configuration du bouton de géolocalisation");
    
    // Rechercher tous les boutons de géolocalisation
    const geolocateButtons = document.querySelectorAll('.geolocation-button');
    logMessage(`Nombre de boutons trouvés: ${geolocateButtons.length}`);
    
    geolocateButtons.forEach(button => {
      logMessage(`Bouton trouvé: ${button.id || 'sans ID'}`);
      
      // Ajouter un gestionnaire d'événements
      button.addEventListener('click', function() {
        logMessage("Clic sur le bouton de géolocalisation");
        
        // Planifier des corrections périodiques après le clic
        setTimeout(fixExistingDistances, 5000);  // 5 secondes
        setTimeout(fixExistingDistances, 10000); // 10 secondes
        setTimeout(fixExistingDistances, 15000); // 15 secondes
      });
      
      logMessage(`Gestionnaire d'événements ajouté au bouton: ${button.id || 'sans ID'}`);
    });
  }
  
  // Initialisation
  function init() {
    logMessage("Initialisation du script d'affichage des distances");
    
    // Intercepter les appels fetch
    interceptFetchCalls();
    
    // Observer les modifications du DOM
    observeDOMChanges();
    
    // Configurer le bouton de géolocalisation
    setupGeolocationButton();
    
    // Corriger les distances existantes
    setTimeout(fixExistingDistances, 1000);
    
    logMessage("Initialisation terminée");
    
    // Afficher un message visible pour confirmer que le script est chargé
    const message = document.createElement('div');
    message.style.position = 'fixed';
    message.style.bottom = '10px';
    message.style.right = '10px';
    message.style.backgroundColor = '#4CAF50';
    message.style.color = 'white';
    message.style.padding = '10px';
    message.style.borderRadius = '5px';
    message.style.zIndex = '9999';
    message.textContent = 'Script d\'affichage des distances chargé';
    document.body.appendChild(message);
    
    // Supprimer le message après 3 secondes
    setTimeout(() => {
      message.remove();
    }, 3000);
  }
  
  // Démarrer l'initialisation
  init();
});
